/*:
 * @target MZ
 * @plugindesc ピクチャをフリックした際の角度に応じてコモンイベントを発生させるプラグイン
 * @author あなたの名前
 *
 * @command enableFlick
 * @text ピクチャのフリックを有効化
 * @desc ピクチャのフリックを有効にし、角度に応じたコモンイベントを設定します。
 *
 * @arg pictureId
 * @type number
 * @min 1
 * @text ピクチャID
 * @desc フリックを有効にするピクチャのIDを指定します。
 *
 * @arg commonEventId
 * @type common_event
 * @text コモンイベント
 * @desc フリック時に発生させるコモンイベントを指定します。
 *
 * @arg minAngle
 * @type number
 * @min 0
 * @max 360
 * @default 0
 * @text 最小角度
 * @desc フリック角度の範囲の最小角度を指定します。
 *
 * @arg maxAngle
 * @type number
 * @min 0
 * @max 360
 * @default 360
 * @text 最大角度
 * @desc フリック角度の範囲の最大角度を指定します。
 *
 * @arg angleVariableId
 * @type variable
 * @text 角度を保存する変数
 * @desc フリックした際の角度を保存する変数を指定します。
 *
 * @help
 * このプラグインは、指定したピクチャをフリックした際に、フリックの角度に
 * 応じて指定したコモンイベントを発生させることができます。
 *
 * 使用方法:
 * 1. プラグインコマンド「ピクチャのフリックを有効化」でピクチャIDとコモンイベントを設定します。
 * 2. ピクチャをフリックすると、指定した角度の範囲内ならコモンイベントが実行されます。
 * 3. 角度は指定した変数に保存されます。
 */

(() => {
    const pluginName = 'PictureFlickPlugin';

    let flickSettings = {};

    // プラグインコマンドの登録
    PluginManager.registerCommand(pluginName, 'enableFlick', args => {
        const pictureId = Number(args.pictureId);
        const commonEventId = Number(args.commonEventId);
        const minAngle = Number(args.minAngle);
        const maxAngle = Number(args.maxAngle);
        const angleVariableId = Number(args.angleVariableId);

        flickSettings[pictureId] = {
            commonEventId: commonEventId,
            minAngle: minAngle,
            maxAngle: maxAngle,
            angleVariableId: angleVariableId
        };
    });

    // マウス操作によるフリック判定
    let startX = 0, startY = 0, isFlicking = false, flickedPictureId = null;

    const calculateFlickAngle = (startX, startY, endX, endY) => {
        const dx = endX - startX;
        const dy = endY - startY;
        const angle = Math.atan2(dy, dx) * (180 / Math.PI);
        return (angle + 360) % 360;  // 角度を0-360に正規化
    };

    const onPictureFlicked = (pictureId, angle) => {
        const settings = flickSettings[pictureId];
        if (!settings) return;

        // 角度を変数に保存
        $gameVariables.setValue(settings.angleVariableId, angle);

        // 角度が範囲内か確認
        if (angle >= settings.minAngle && angle <= settings.maxAngle) {
            // コモンイベントの実行
            $gameTemp.reserveCommonEvent(settings.commonEventId);
        }
    };

    const _TouchInput_onMouseDown = TouchInput._onMouseDown;
    TouchInput._onMouseDown = function(event) {
        _TouchInput_onMouseDown.call(this, event);
        const pictureId = getPictureUnderMouse();
        if (pictureId) {
            startX = event.pageX;
            startY = event.pageY;
            isFlicking = true;
            flickedPictureId = pictureId;
        }
    };

    const _TouchInput_onMouseUp = TouchInput._onMouseUp;
    TouchInput._onMouseUp = function(event) {
        _TouchInput_onMouseUp.call(this, event);
        if (isFlicking && flickedPictureId) {
            const endX = event.pageX;
            const endY = event.pageY;
            const angle = calculateFlickAngle(startX, startY, endX, endY);
            onPictureFlicked(flickedPictureId, angle);
            isFlicking = false;
            flickedPictureId = null;
        }
    };

    // ピクチャの座標にマウスがあるか確認する関数
    function getPictureUnderMouse() {
        for (let i = $gameScreen._pictures.length - 1; i >= 1; i--) {
            const picture = $gameScreen.picture(i);
            if (picture && isMouseOverPicture(picture)) {
                return i;
            }
        }
        return null;
    }

    // マウスがピクチャの範囲内にあるか判定する
    function isMouseOverPicture(picture) {
        const bitmap = ImageManager.loadPicture(picture.name());
        const mouseX = TouchInput.x;
        const mouseY = TouchInput.y;
        const pictureX = picture.x();
        const pictureY = picture.y();
        const width = bitmap.width;
        const height = bitmap.height;
        return mouseX >= pictureX && mouseX <= pictureX + width &&
               mouseY >= pictureY && mouseY <= pictureY + height;
    }

})();
